# ESP32 Touch Dimmer component for ESPHome
# Copyright (C) 2025 Sybren A. Stüvel
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import esphome.codegen as cg
import esphome.config_validation as cv
from esphome.components import light, binary_sensor, script
from esphome.const import (
    CONF_ID,
    CONF_LIGHT_ID,
)

CODEOWNERS = ["@sybrenstuvel"]

CONF_TOUCH_ID = "touch_id"
CONF_STEP_SIZE = "step_size"

AUTO_LOAD = ["binary_sensor", "light"]
REQUIRES = ["script"]
MULTI_CONF = True

touch_dimmer_ns = cg.esphome_ns.namespace("touch_dimmer")
TouchDimmer = touch_dimmer_ns.class_("TouchDimmer", cg.Component)

CONFIG_SCHEMA = cv.Schema(
    {
        cv.GenerateID(): cv.declare_id(TouchDimmer),
        cv.Required(CONF_LIGHT_ID): cv.use_id(light.LightState),
        cv.Required(CONF_TOUCH_ID): cv.use_id(binary_sensor.BinarySensor),
        cv.Optional(CONF_STEP_SIZE, default=0.01): cv.positive_float,
    }
).extend(cv.COMPONENT_SCHEMA)


async def to_code(config):
    var = cg.new_Pvariable(config[CONF_ID])
    await cg.register_component(var, config)

    # Let the component know its own ID, so that it can generate sub-components
    # with this ID as their prefix.
    cg.add(var.set_id(config[CONF_ID].id))

    light_var = await cg.get_variable(config[CONF_LIGHT_ID])
    cg.add(var.set_light(light_var))

    touch_var = await cg.get_variable(config[CONF_TOUCH_ID])
    cg.add(var.set_touch_sensor(touch_var))

    cg.add(var.set_step_size(config[CONF_STEP_SIZE]))
