#pragma once

#include "esphome/components/binary_sensor/automation.h"
#include "esphome/components/binary_sensor/binary_sensor.h"
#include "esphome/components/esp32_touch/esp32_touch.h"
#include "esphome/components/light/automation.h"
#include "esphome/components/script/script.h"
#include "esphome/core/application.h"
#include "esphome/core/automation.h"
#include "esphome/core/base_automation.h"
#include "esphome/core/component.h"
#include "esphome/core/log.h"
#include <string>

using namespace esphome;

using ScriptExecuteAction = script::ScriptExecuteAction<script::Script<>>;
using ScriptStopAction = script::ScriptStopAction<script::SingleScript<>>;

namespace esphome {
namespace touch_dimmer {

class TouchDimmer : public Component {
  static constexpr uint64_t dimStartDelayMs = 350;
  static constexpr uint64_t dimPeriodMs = 20;
  static constexpr uint32_t toggleTimeThresholdMs = 3000;
  static constexpr auto LOGTAG = "TouchDimmer";

  float dimming_dir = NAN;
  uint32_t lastActionMs = 0;

  std::string my_id_;
  std::string my_id_dimming_start_;
  std::string my_id_dimming_stop_;
  light::LightState *light_{nullptr};
  binary_sensor::BinarySensor *touch_sensor_{nullptr};
  float step_size_{0.01f};

  script::SingleScript<> *scriptDimStart;
  script::SingleScript<> *scriptDimStop;

 public:
  void set_id(const std::string &id)
  {
    this->my_id_ = id;
    this->my_id_dimming_start_ = this->my_id_ + "__script_dimming_start";
    this->my_id_dimming_stop_ = this->my_id_ + "__script_dimming_stop";
  }
  void set_light(light::LightState *light)
  {
    this->light_ = light;
  }
  void set_touch_sensor(binary_sensor::BinarySensor *touch_sensor)
  {
    this->touch_sensor_ = touch_sensor;
  }
  void set_step_size(const float step_size)
  {
    this->step_size_ = step_size;
  }

  void setup() override
  {
    ESP_LOGD(LOGTAG, "Setting up TouchDimmer component...");
    if (!touch_sensor_) {
      ESP_LOGE(LOGTAG, "No touch sensor configured.");
      this->mark_failed();
      return;
    }
    if (!light_) {
      ESP_LOGE(LOGTAG, "No light configured.");
      this->mark_failed();
      return;
    }

    touch_sensor_->add_on_state_callback([this](bool state) { this->on_touch(state); });

    this->createDimStartScript();
    this->createDimStopScript();
    this->setupBinarySensor();
  }

  void loop() override
  {
    // Optional: do periodic logic here
  }

 protected:
  void on_touch(bool state)
  {
    ESP_LOGD(LOGTAG, "Touch sensor state: %s", state ? "ON" : "OFF");
    // You could start/stop dimming here based on `state`
    // For now just toggles the light for demo
    if (state && light_ != nullptr) {
      light_->toggle();
    }
  }

 private:
  // Dimming start script, to be called from the 'on_press' handler of the touch pad:
  void createDimStartScript()
  {
    // Wait to see if the 'touch' lasts for long enough:
    auto *startDelayAction = new DelayAction<>();
    startDelayAction->set_delay(dimStartDelayMs + 50);  // Add a little margin.
    App.register_component(startDelayAction);

    // Toggle the dimming direction:
    LambdaAction<> *toggleDimDirLambda = new LambdaAction<>([&]() -> void {
      const float cur_bright = this->light_->current_values.get_brightness();
      const bool is_on = this->light_->current_values.is_on();

      const uint32_t currentTime = millis();
      const uint32_t diffTime = currentTime - this->lastActionMs;
      ESP_LOGD(LOGTAG, "toggleDimDirLambda: now = %ul  diff=%ul", currentTime, diffTime);
      this->lastActionMs = currentTime;

      // this->dimming_dir = NAN;
      if (diffTime > toggleTimeThresholdMs || std::isnan(this->dimming_dir)) {
        // Dim towards the furthest extreme.
        ESP_LOGD(LOGTAG, "dimming towards furthest extreme");

        if (!is_on || cur_bright < 0.5f) {
          this->dimming_dir = 1;
        }
        else {
          this->dimming_dir = -1;
        }
      }
      else {
        // Just toggle the dimming direction from the last time.
        ESP_LOGD(LOGTAG, "dimming_dir is known, toggling it");
        this->dimming_dir *= -1;
      }

      if (!is_on) {
        // Once we start increasing the brightness, it should start at 0%.
        this->light_->current_values.set_brightness(0.0f);
      }
    });

    // Perform a dim step every `dimPeriodMs` milliseconds:
    auto *dimDelayAction = new DelayAction<>();
    dimDelayAction->set_delay(dimPeriodMs);
    App.register_component(dimDelayAction);

    // Do the actual dimming:
    LambdaAction<> *dimStepLambda = new LambdaAction<>([&]() -> void {
      if (std::isnan(dimming_dir)) {
        ESP_LOGE(LOGTAG,
                 "%s: dimming_dir is NaN, no idea which direction to go into",
                 this->my_id_.c_str());
        return;
      }

      const float step = this->dimming_dir * this->step_size_;
      const float cur_bright = this->light_->current_values.get_brightness();
      const float new_bright = clamp(cur_bright + step, 0.0f, 1.0f);

      // ESP_LOGI(LOGTAG,
      //          "%s: dimming_dir=%.0f  step=%.3f  cur_bright=%.3f  new_bright=%.3f",
      //          this->my_id_.c_str(),
      //          this->dimming_dir,
      //          step,
      //          cur_bright,
      //          new_bright);

      {
        auto call = this->light_->make_call();
        call.set_brightness(new_bright);
        call.set_transition_length(dimPeriodMs / 2.0f);
        call.set_state(new_bright > 0);
        call.perform();
      }

      if (new_bright <= 0.0f || new_bright >= 0.999f) {
        this->scriptDimStop->execute();
      }

      this->lastActionMs = millis();
    });

    // Construct the while-loop:
    auto *whileCondition = new binary_sensor::BinarySensorCondition<>(this->touch_sensor_, true);
    auto *whileAction = new WhileAction<>(whileCondition);
    whileAction->add_then({dimDelayAction, dimStepLambda});

    // Construct the script:
    scriptDimStart = new script::SingleScript<>();
    scriptDimStart->set_name(LOG_STR(this->my_id_dimming_start_.c_str()));

    auto *dimStartAutom = new Automation<>(scriptDimStart);
    dimStartAutom->add_actions({startDelayAction, toggleDimDirLambda, whileAction});
  }

  void createDimStopScript()
  {
    // Stop the dimming script:
    auto *scriptStopAction = new ScriptStopAction(scriptDimStart);

    // Construct the script:
    scriptDimStop = new script::SingleScript<>();
    scriptDimStop->set_name(LOG_STR(this->my_id_dimming_stop_.c_str()));

    auto *dimStopAutom = new Automation<>(scriptDimStop);
    dimStopAutom->add_actions({scriptStopAction});
  }

  void setupBinarySensor()
  {
    auto *pressTrigger = new binary_sensor::PressTrigger(this->touch_sensor_);
    auto *pressScriptAction = new ScriptExecuteAction(scriptDimStart);
    auto *pressAutomation = new Automation<>(pressTrigger);
    pressAutomation->add_actions({pressScriptAction});

    auto *releaseTrigger = new binary_sensor::ReleaseTrigger(this->touch_sensor_);
    auto *releaseScriptAction = new ScriptExecuteAction(scriptDimStop);
    auto *releaseAutomation = new Automation<>(releaseTrigger);
    releaseAutomation->add_actions({releaseScriptAction});

    auto *clickTrigger = new binary_sensor::ClickTrigger(this->touch_sensor_, 25, dimStartDelayMs);
    auto *lightToggleAction = new light::ToggleAction<>(this->light_);
    LambdaAction<> *clearDimDirectionLambda = new LambdaAction<>([&]() -> void {
      ESP_LOGD(LOGTAG, "clearDimDirectionLambda");
      this->dimming_dir = NAN;
    });

    auto *clickAutomation = new Automation<>(clickTrigger);
    clickAutomation->add_actions({lightToggleAction, clearDimDirectionLambda});
  }
};

}  // namespace touch_dimmer
}  // namespace esphome
